import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Info, Check, Calendar, Clock, MapPin, Phone, Mail } from 'lucide-react'
import './BookTraining.css'

const BookTraining = () => {
  const navigate = useNavigate()
  const [selectedType, setSelectedType] = useState(null)
  const [selectedDate, setSelectedDate] = useState(null)
  const [selectedTime, setSelectedTime] = useState(null)

  const trainingTypes = [
    { id: 'yoga', name: 'Йога', emoji: '🧘', gradient: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)' },
    { id: 'crossfit', name: 'Кроссфит', emoji: '💪', gradient: 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)' },
    { id: 'pilates', name: 'Пилатес', emoji: '🤸', gradient: 'linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)' },
    { id: 'stretching', name: 'Стретчинг', emoji: '🧘‍♀️', gradient: 'linear-gradient(135deg, #fa709a 0%, #fee140 100%)' }
  ]

  const availableDates = [
    '2025-03-20',
    '2025-03-21',
    '2025-03-22',
    '2025-03-23',
    '2025-03-24',
    '2025-03-25',
    '2025-03-26'
  ]

  const availableTimes = [
    '09:00', '10:00', '11:00', '12:00',
    '17:00', '18:00', '19:00', '20:00'
  ]

  const trainers = {
    'yoga': 'Анна Петрова',
    'crossfit': 'Иван Сидоров',
    'pilates': 'Мария Козлова',
    'stretching': 'Ольга Новикова'
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedType && selectedDate && selectedTime) {
      const savedBookings = localStorage.getItem('fitnessBookings')
      const bookings = savedBookings ? JSON.parse(savedBookings) : { active: [], past: [] }
      
      const newBooking = {
        id: Date.now(),
        type: trainingTypes.find(t => t.id === selectedType).name,
        date: selectedDate,
        time: selectedTime,
        trainer: trainers[selectedType] || 'Тренер',
        status: 'active'
      }

      bookings.active.push(newBooking)
      localStorage.setItem('fitnessBookings', JSON.stringify(bookings))
      
      setSelectedType(null)
      setSelectedDate(null)
      setSelectedTime(null)
      
      navigate('/')
    }
  }

  const advantages = [
    'Персональный подход к каждому клиенту',
    'Современное оборудование премиум-класса',
    'Гибкий график работы с 6:00 до 23:00',
    'Профессиональные тренеры с сертификатами',
    'Индивидуальные программы тренировок',
    'Зона отдыха и восстановления'
  ]

  return (
    <div className="book-training-page">
      <div className="page-intro">
        <h1 className="intro-title">Запись на тренировку</h1>
        <p className="intro-subtitle">Выберите удобное время и начните свой путь к идеальной форме</p>
      </div>

      <div className="advantages-section">
        <h2 className="advantages-title">
          <Info className="title-icon" size={24} />
          Преимущества нашего фитнес-центра
        </h2>
        <div className="advantages-list">
          {advantages.map((advantage, index) => (
            <div key={index} className="advantage-item">
              <Check className="check-mark" size={20} />
              <span>{advantage}</span>
            </div>
          ))}
        </div>
      </div>

      <form onSubmit={handleSubmit} className="booking-form">
        <div className="form-section">
          <h2 className="section-title">
            <span className="title-emoji">💪</span>
            Вид тренировки
          </h2>
          <div className="training-types-grid">
            {trainingTypes.map((type) => (
              <button
                key={type.id}
                type="button"
                className={`type-card ${selectedType === type.id ? 'selected' : ''}`}
                onClick={() => setSelectedType(type.id)}
                style={{
                  background: selectedType === type.id ? type.gradient : 'var(--bg-light)'
                }}
              >
                <span className="type-emoji">{type.emoji}</span>
                <span className="type-name">{type.name}</span>
                {selectedType === type.id && (
                  <div className="check-overlay">
                    <Check className="check-icon" size={20} />
                  </div>
                )}
              </button>
            ))}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Calendar className="section-icon" size={24} />
            Дата
          </h2>
          <div className="dates-scroll">
            {availableDates.map((date) => {
              const dateObj = new Date(date)
              const isSelected = selectedDate === date
              const isToday = date === new Date().toISOString().split('T')[0]
              return (
                <button
                  key={date}
                  type="button"
                  className={`date-button ${isSelected ? 'selected' : ''} ${isToday ? 'today' : ''}`}
                  onClick={() => setSelectedDate(date)}
                >
                  <div className="date-day">{dateObj.toLocaleDateString('ru-RU', { day: 'numeric' })}</div>
                  <div className="date-weekday">{dateObj.toLocaleDateString('ru-RU', { weekday: 'short' })}</div>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Clock className="section-icon" size={24} />
            Время
          </h2>
          <div className="times-grid">
            {availableTimes.map((time) => (
              <button
                key={time}
                type="button"
                className={`time-button ${selectedTime === time ? 'selected' : ''}`}
                onClick={() => setSelectedTime(time)}
              >
                {time}
              </button>
            ))}
          </div>
        </div>

        <button
          type="submit"
          className="submit-button"
          disabled={!selectedType || !selectedDate || !selectedTime}
        >
          Записаться
        </button>
      </form>

      <div className="info-section">
        <h2 className="info-title">Как нас найти</h2>
        <div className="info-grid">
          <div className="info-card">
            <MapPin className="info-card-icon" size={24} />
            <div>
              <h3>Адрес</h3>
              <p>г. Санкт-Петербург,<br />Невский проспект, д. 25</p>
            </div>
          </div>
          <div className="info-card">
            <Phone className="info-card-icon" size={24} />
            <div>
              <h3>Телефон</h3>
              <p>+7 (812) 234-56-78</p>
            </div>
          </div>
          <div className="info-card">
            <Mail className="info-card-icon" size={24} />
            <div>
              <h3>Email</h3>
              <p>contact@fitzone.ru</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

export default BookTraining

